#!/bin/bash
set -euo pipefail

#=============================================================================
# Script: 3-configure-web.sh
# PURPOSE: Configure nginx web server and content serving
#=============================================================================

# HTTP Landing Page
server {
listen 80 default_server;
server_name _;
root /var/www/landing;
index index.html;

location ~* ^/(generate_204|gen_204|.*captive.*|connecttest.*|ncsi.*|success\.html|hotspot.*)$ {
alias /var/www/landing/index.html;
add_header Content-Type text/html;
}

location = /success {
return 302 https://rapturekit.com/rapture_kit/;
}

location /health { return 200 "RKPi5 HTTP OK"; add_header Content-Type text/plain; access_log off; }

location / { try_files $uri $uri/ /index.html =404; }
}

# HTTPS Portal
server {
listen 443 ssl http2 default_server;
server_name rapturekit.com www.rapturekit.com _;

ssl_certificate /etc/ssl/RKPi5/certificate.crt;
ssl_certificate_key /etc/ssl/RKPi5/private.key;
ssl_protocols TLSv1.2 TLSv1.3;
ssl_ciphers ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:HIGH:!aNULL:!MD5;
ssl_prefer_server_ciphers on;

add_header X-Frame-Options DENY;
add_header X-Content-Type-Options nosniff;
add_header X-XSS-Protection "1; mode=block";
add_header Strict-Transport-Security "max-age=31536000" always;

location /rapture_kit/ {
alias ${CONTENT_DIR}/;
try_files $uri $uri/ /index.html =404;
location ~* \.(css|js|png|jpg|jpeg|gif|ico|svg|webp)$ { expires 1y; add_header Cache-Control "public, immutable"; access_log off; }
location ~* \.(mp4|webm|mp3|ogg|avi|mov|wav)$ { expires 30d; add_header Cache-Control "public"; add_header Accept-Ranges bytes; }
location ~* \.(pdf|doc|docx|zip|txt)$ { expires 7d; add_header Cache-Control "public"; }
}

location = / { return 301 https://$host/rapture_kit/; }

location /admin/ {
alias /var/www/admin/;
try_files $uri $uri/ /index.html =404;
auth_basic "RKPi5 Admin";
auth_basic_user_file /etc/nginx/.htpasswd;
}

location /health { return 200 "RKPi5 HTTPS OK"; add_header Content-Type text/plain; access_log off; }
}
EOF

sudo mkdir -p /var/www/landing
sudo tee /var/www/landing/index.html >/dev/null <<'HTML'
<!DOCTYPE html><html lang="en"><head><meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1"><title>Rapture Kit Training Portal</title>
<style>*{box-sizing:border-box}body{font-family:Segoe UI,Arial;background:linear-gradient(135deg,#667eea,#764ba2);min-height:100vh;display:flex;align-items:center;justify-content:center;color:#fff;margin:0}.c{max-width:640px;background:rgba(255,255,255,.1);padding:36px;border-radius:16px;backdrop-filter:blur(8px);text-align:center}.btn{background:#28a745;color:#fff;padding:14px 24px;border-radius:10px;text-decoration:none;display:inline-block;margin-top:16px}</style>
</head><body><div class="c">
<h1>🎓 Rapture Kit Training</h1><p>Welcome to your offline training system!</p>
<p><small>You may see a security warning next — choose <b>Advanced</b> → <b>Proceed</b>.</small></p>
<p><a class="btn" href="http://10.42.0.1/success">🚀 Enter Training Portal</a></p>
</div></body></html>
HTML

sudo tee /var/www/admin/index.html >/dev/null <<HTML
<!DOCTYPE html><html><head><meta name="viewport" content="width=device-width, initial-scale=1"><title>RKPi5 Admin</title>
<style>body{font-family:Arial;background:#f5f5f5;margin:20px}.card{background:#fff;padding:20px;border-radius:10px;box-shadow:0 2px 5px rgba(0,0,0,.1);margin:16px 0}.btn{background:#007cba;color:#fff;padding:10px 16px;border-radius:6px;text-decoration:none}</style></head>
<body><h1>🎓 RKPi5 v2.1 Admin</h1>
<div class="card">Content: <code>${CONTENT_DIR}</code><br>Portal: <a href="/rapture_kit/" class="btn">Open</a></div>
<div class="card"><a class="btn" href="/health">Health Check</a></div>
</body></html>
HTML

sudo nginx -t && sudo systemctl restart nginx

echo "✅ Web configuration complete (serving from ${CONTENT_DIR})"
